/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.ui.dialogs;

import com.jniwrapper.win32.ui.Wnd;
import com.jniwrapper.win32.shell.Shell32;
import com.jniwrapper.win32.shell.ShellIcon;
import com.jniwrapper.win32.system.Kernel32;
import com.jniwrapper.*;

import java.io.File;
import java.awt.Window;

/**
 * This class provides functionality that allows to display  a dialog box that allows the user to choose an icon
 * from the selection available embedded in a resource such as an executable or DLL file
 *
 * @author Vladimir Kondrashchenko
 */
public class SelectIconDialog
{
    private static final String FUNCTION_PICK_ICON_DIALOG = "PickIconDlg";

    private static final String DEFAULT_ICON_FILE_NAME = Kernel32.getSystemDirectory() + "shell32.dll";
    private File _iconFile = new File(DEFAULT_ICON_FILE_NAME);
    private int _iconIndex = 0;
    private Window _owner = null;

    /**
     * Creates instance of the <code>SelectIconDialog</code>
     */
    public SelectIconDialog()
    {

    }

    /**
     * Creates instance of the <code>SelectIconDialog</code> initialized by the file that contains icons
     * @param iconFile The file that contains icons
     */
    public SelectIconDialog(File iconFile)
    {
        setIconFile(iconFile);
    }

    /**
     * Creates instance of the <code>SelectIconDialog</code> initialized by the file that contains icons
     *
     * @param iconFile The file that contains icons
     * @param iconIndex Index of the icon in the file specified by the <code>iconFile</code> parameter
     *  that will be selected by default
     */
    public SelectIconDialog(File iconFile, int iconIndex)
    {
        setIconFile(iconFile);
        setIconIndex(iconIndex);
    }

    /**
     * Creates instance of the <code>SelectIconDialog</code>
     *
     * @param owner Parent window for the dialog
     */
    public SelectIconDialog(Window owner)
    {
        setOwner(owner);
    }

    /**
     * Creates instance of the <code>SelectIconDialog</code> initialized by the file that contains icons
     *
     * @param owner Parent window for the dialog
     * @param iconFile The file that contains icons
     */
    public SelectIconDialog(Window owner, File iconFile)
    {
        setOwner(owner);
        setIconFile(iconFile);
    }

    /**
     * Creates instance of the <code>SelectIconDialog</code> initialized by the file that contains icons
     *
     * @param owner Parent window for the dialog
     * @param iconFile The file that contains icons
     * @param iconIndex Index of the icon in the file specified by the <code>iconFile</code> parameter
     *  that will be selected by default
     */
    public SelectIconDialog(Window owner, File iconFile, int iconIndex)
    {
        setOwner(owner);
        setIconFile(iconFile);
        setIconIndex(iconIndex);
    }

    /**
     * Sets value that specifies file with icons that will be displayed in the dialog
     *
     * @param value Value that specifies file with icons that will be displayed in the dialog
     */
    public void setIconFile(File value)
    {
        if (value.isFile())
        {
            _iconFile = value;
        }
        else
        {
            _iconFile = new File(DEFAULT_ICON_FILE_NAME);
        }
    }

    /**
     * Returns value that specifies file with icons that will be displayed in the dialog
     *
     * @return File with icons
     */
    public File getIconFile()
    {
        return _iconFile;
    }

    /**
     * Sets index of the selected icon
     *
     * @param value Index of the selected icon
     */
    public void setIconIndex(int value)
    {
        _iconIndex = value;
    }

    /**
     * Returns index of the selected icon
     *
     * @return Index of the selected icon
     */
    public int getIconIndex()
    {
        return _iconIndex;
    }

    /**
     * Returns selected icon
     *
     * @return Selected icon
     */
    public ShellIcon getShellIcon()
    {
        if (_iconFile.exists())
            return new ShellIcon(_iconFile, _iconIndex);
        else
            return null;
    }

    /**
     * Sets the parent window for the dialog
     *
     * @param value The parent window for the dialog
     */
    public void setOwner(Window value)
    {
        _owner = value;
    }

    /**
     * Returns the parent window for the dialog.
     *
     * @return The parent window for the dialog.
     */
    public Window getOwner()
    {
        return _owner;
    }

    /**
     * Displays dialog box
     *
     * @return true if the icon was selected. Otherwise false.
     */
    public boolean execute()
    {
        Wnd owner = _owner != null ? new Wnd(_owner) : new Wnd();

        final Function pickIconDialog = Shell32.getInstance().getFunction(FUNCTION_PICK_ICON_DIALOG);

        Int result = new Int();
        Str sFileName = new Str();
        sFileName.setValue(getIconFile().getAbsolutePath());
        LongInt nBuf = new LongInt(sFileName.getMaxLength());
        LongInt nIconIndex = new LongInt(getIconIndex());

        if (_owner != null)
            DialogHelper.invokeDialog(_owner, pickIconDialog, result, new Parameter[]{owner, new Pointer(sFileName),
                                                                                      nBuf, new Pointer(nIconIndex)});
        else
            pickIconDialog.invoke(result, owner, new Pointer(sFileName), nBuf, new Pointer(nIconIndex));

        if (result.getValue() != 0)
        {
            setIconFile(new File(sFileName.getValue()));
            setIconIndex((int)nIconIndex.getValue());
            if (getIconFile().isFile())
                return true;
            else
                return false;
        }
        else
        {
            return false;
        }
    }
}