/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.system.eventlog;

import com.jniwrapper.util.EnumItem;

import java.util.Date;

/**
 * This class provides functionality for working with event log messages.
 *
 * @author Vladimir Kondrashchenko
 * @see EventLog
 */
public class EventLogMessage
{
    private int _recordNumber;
    private long _eventID = 0;
    private Type _eventType;
    private int _category = 0;
    private Date _date;
    private byte[] _data;
    private String _source;
    private String _user;
    private String _computer;
    private String _message;

    EventLogMessage()
    {
    }

    /**
     * Creates event log message from the specified source
     *
     * @param sourceName The name of the source of the event log message
     */
    public EventLogMessage(String sourceName)
    {
        if (sourceName == null)
        {
            throw new IllegalArgumentException("Illegal message source.");
        }
        _source = sourceName;
    }

    /**
     * Sets identifier of the event log message
     *
     * @param eventID Identifier of the event log message
     */
    public void setEventID(long eventID)
    {
        if (eventID < 0)
        {
            throw new IllegalArgumentException("Event ID must be positive.");
        }
        _eventID = eventID;
    }

    /**
     * Sets type of the event log message
     *
     * @param eventType The type of the event log message
     */
    public void setEventType(Type eventType)
    {
        _eventType = eventType;
    }

    /**
     * Sets category of the event log message
     *
     * @param category Category of the event log message
     */
    public void setCategory(int category)
    {
        if (category < 0)
        {
            throw new IllegalArgumentException("Category argument must be positive.");
        }
        _category = category;
    }

    /**
     * Sets the date of the event log message occurrence 
     *
     * @param date The date of the event log message occurrence
     */
    void setDate(Date date)
    {
        _date = date;
    }

    /**
     * Sets the date of the event log message occurrence
     *
     * @param data The date of the event log message occurrence
     */
    public void setData(byte[] data)
    {
        _data = data;
    }

    /**
     * Sets the source from which event has occurred
     *
      * @param source The source from which event has occurred
     */
    public void setSource(String source)
    {
        _source = source;
    }

    /**
     * Sets the name of the user
     *
      * @param user The name of the user
     */
    void setUser(String user)
    {
        _user = user;
    }

    /**
     * Sets computer name
     *
     * @param computer The name of the computer
     */
    void setComputer(String computer)
    {
        _computer = computer;
    }

    /**
     * Sets message text
     *
     * @param message Message text
     */
    public void setMessage(String message)
    {
        _message = message;
    }

    /**
     * Sets the number of the record
     *
     * @param recordNumber The number of the record
     */
    void setRecordNumber(int recordNumber)
    {
        _recordNumber = recordNumber;
    }

    /**
     * Returns identifier of the event log message
     *
     * @return Identifier of the event log message
     */
    public long getEventID()
    {
        return _eventID;
    }

    /**
     * Returns event log message type
     *
     * @return The type of the event log message
     */
    public Type getEventType()
    {
        return _eventType;
    }

    /**
     * Returns event log message category
     *
     * @return The category of the event log message
     */
    public int getCategory()
    {
        return _category;
    }

    /**
     * Returns event log message date
     *
     * @return The date of the event log message
     */
    public Date getDate()
    {
        return _date;
    }

    /**
     * Returns event log message date
     *
     * @return The date of the event log message
     */
    public byte[] getData()
    {
        return _data;
    }

    /**
     * Returns event log message source
     *
     * @return The source of the event log message
     */
    public String getSource()
    {
        return _source;
    }

    /**
     * Returns user name
     *
     * @return The name of the user
     */
    public String getUser()
    {
        return _user;
    }

    /**
    * Returns computer name
    *
    * @return The computer name
    */
    public String getComputer()
    {
        return _computer;
    }

    /**
     * Returns event log message
     *
     * @return The event log message
     */
    public String getMessage()
    {
        return _message;
    }

    /**
     * Returns event log message number
     *
     * @return The number of the event log message
     */
    public int getRecordNumber()
    {
        return _recordNumber;
    }

    /**
     * Specifies the type of an event log message.
     */
    public static class Type extends EnumItem
    {
        private String _name;

        /**
         * Information event.
         */
        public static final Type SUCCESS = new Type(0x0000);
        /**
         * Error event.
         */
        public static final Type ERROR = new Type(0x0001);
        /**
         * Warning event.
         */
        public static final Type WARNING = new Type(0x0002);
        /**
         * Information event.
         */
        public static final Type INFORMATION = new Type(0x0004);
        /**
         * Success Audit event.
         */
        public static final Type AUDIT_SUCCESS = new Type(0x0008);
        /**
         * Failure Audit event.
         */
        public static final Type AUDIT_FAILURE = new Type(0x0010);

        Type(int value)
        {
            super(value);
            switch (value)
            {
                case 0x0000 :
                    _name = "Success";
                    break;
                case 0x0001 :
                    _name = "Error";
                    break;
                case 0x0002 :
                    _name = "Warning";
                    break;
                case 0x0004 :
                    _name = "Information";
                    break;
                case 0x0008 :
                    _name = "Success Audit";
                    break;
                case 0x0010 :
                    _name = "Failure Audit";
                    break;
            }
        }

        /**
         * Returns event log message type name.
         *
         * @return The name of the type of an event log message.
         */
        public String toString()
        {
            return _name;
        }
    }
}