/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32;

import com.jniwrapper.win32.system.Kernel32;

/**
 * This class stores general Windows API function name and returns the
 * encoding-specific name on demand.
 *
 * @author Serge Piletsky
 */
public class FunctionName
{
    static final char SUFFIX_ANSI = 'A';
    static final char SUFFIX_UNICODE = 'W';

    private String _name;
    private boolean _hasANSISuffix = true;

    private static boolean _useUnicode = Kernel32.getInstance().isUnicode();

    /**
     * Sets the value of the property that specifies whether to use Unicode ore ANSI functions names
     * @param value true if the Unicode functions names must be used. Otherwise false.
     */
    public static void useUnicodeNames(boolean value)
    {
        _useUnicode = value;
    }

    /**
     * Specifies whether the Unicode functions name is used.
     * @return true if the Unicode functions names is used. Otherwise false.
     */
    public static boolean usesUnicodeNames()
    {
        return _useUnicode;
    }

    /**
     * Creates instance of the <code>FunctionName</code>
     *
     * @param name The name of the function. This name will be converted to the name that correspond
     * to the Unicode or ANSI function name depending on system requirements
     */
    public FunctionName(String name)
    {
        _name = name;
    }

    /**
     * Creates instance of the <code>FunctionName</code>
     *
     * @param name The name of the function. This name will be converted to the name that correspond
     * to the Unicode or ANSI function name depending on system requirements
     * @param hasANSISuffix true if the function name requires ANSI suffix. Otherwise false
     */
    public FunctionName(String name, boolean hasANSISuffix)
    {
        _name = name;
        _hasANSISuffix = hasANSISuffix;
    }

    /**
     * Specifies whether the function name requires ANSI suffix or not 
     * @return true if the function name requires ANSI suffix. Otherwise false
     */
    public boolean hasANSISuffix()
    {
        return _hasANSISuffix;
    }

    /**
     * Returns the name of the function without suffix
     * @return The name of the function without suffix
     */
    public String getName()
    {
        return _name;
    }

    /**
     * Returns the encoding-specific function name.
     *
     * @param unicode if true, Unicode name will be returned.
     * @return function name with encoding-specific suffix.
     */
    public String getEncodingSpecificName(boolean unicode)
    {
        StringBuffer result = new StringBuffer(getName());
        if (unicode)
        {
            result.append(SUFFIX_UNICODE);
        }
        else if (_hasANSISuffix)
        {
            result.append(SUFFIX_ANSI);
        }
        return result.toString();
    }

    /**
     * Returns encoding-specific name of the function
     * @return Encoding-specific name of the function
     */
    public String toString()
    {
        return getEncodingSpecificName(_useUnicode);
    }
}